/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Collections;
using System.Globalization;
using System.Reflection;
using System.Text;
using Borland.Eco.Interfaces;

namespace Borland.Eco.UmlRt
{
	public abstract class ModelElementSameHash
	{
		private sealed class HashCodeProviderImpl: IHashCodeProvider
		{
			int IHashCodeProvider.GetHashCode(object obj)
			{
				if (obj == null)
					throw new ArgumentNullException("obj"); // do not localize
				return ((IModelElement)obj).GetIsSameHashCode();
			}
		}

		private sealed class ComparerImpl: IComparer
		{
			int IComparer.Compare(object x, object y)
			{
				// Do we want to check x or ex for null?
				if (x == null)
					throw new ArgumentNullException("x"); // do not localize
				if (y == null)
					throw new ArgumentNullException("y"); // do not localize
				IModelElement ex = x as IModelElement;
				IModelElement ey = y as IModelElement;
				return ex.IsSame(ey) ? 0 : 1;
			}
		}

		private readonly static IComparer m_Comparer = new ComparerImpl();
		private readonly static IHashCodeProvider m_HashCodePrivider = new HashCodeProviderImpl();

		public static IComparer Comparer
		{
			get { return m_Comparer; }
		}

		public static IHashCodeProvider HashCodeProvider
		{
			get { return m_HashCodePrivider; }
		}
	}

	public enum ScopeKind
	{
		Instance,
		Classifier
	}
	public enum ChangeableKind
	{
		Frozen,
		Changeable
	}
	public enum OrderingKind
	{
		Ordered,
		Unordered
	}
	public enum AggregationKind
	{
		None,
		Aggregate,
		Composite
	}
	public enum VisibilityKind
	{
		Private_,
		Protected_,
		Public_,
		Package
	}

	public enum ParameterDirectionKind
	{
		In,
		Out,
		InOut
	}

	public enum AssociationEndKind
	{
		Normal,
		ToLinkObject,
		InnerLink
	}
	public enum CollectionTypeKind
	{
		Collection,
		Bag,
		Set,
		Sequence,
		OrderedSet
	}
	public enum ClassifierType
	{
		Class,
		Collection,
		Primitive,
		Enumeration,
		Tuple
	}
	public enum FeatureType
	{
		Attribute,
		AssociationEnd,
		Method
	}
	public enum DeleteAction
	{
		Cascade,
		Allow,
		Prohibit
	}
	public enum SaveAction
	{
		None,
		Freeze,
		DbAssign
	}
	public enum TableMapping
	{
		Own,
		Parent,
		Children,
		Imported
	}
	public interface IModelElementCollection: ICollection
	{
		IModelElement GetItemByName(string name);
		IModelElement this[int index] { get; }
		bool Contains(object value);
	}

	public interface ITaggedValueCollection: ICollection
	{
		ITaggedValue GetItemByTag(string tag);
		ITaggedValue this[int index] { get; }
		string ValueForTag(string tag, string defaultValue);
		string ValueForTag(string tag);
	}

	public interface IConstraintCollection: ICollection
	{
		IConstraint GetItemByName(string name);
		IConstraint this[int index] { get; }
	}

	public interface IStereotypeCollection: ICollection
	{
		IStereotype GetItemByName(string name);
		IStereotype this[int index] { get; }
	}

	public interface IQualifierCollection: ICollection
	{
		IQualifier this[int index] { get; }
	}

	public interface IEnumerationLiteralCollection: ICollection
	{
		IEnumerationLiteral this[int index] { get; }
	}

	public interface IClassifierCollection: ICollection
	{
		IClassifier GetItemByName(string name);
		IClassifier this[int index] { get; }
	}

	public interface IClassCollection: ICollection
	{
		IClass GetItemByName(string name);
		IClass this[int index] { get; }
	}

	public interface IPackageCollection: ICollection
	{
		IPackage GetItemByName(string name);
		IPackage this[int index] { get; }
	}

	public interface IAssociationCollection: ICollection
	{
		IAssociation GetItemByName(string name);
		IAssociation this[int index] { get; }
	}

	public interface IFeatureCollection: ICollection
	{
		IFeature GetItemByName(string name);
		IFeature this[int index] { get; }
	}

	public interface IParameterCollection: ICollection
	{
		IParameter GetItemByName(string name);
		IParameter this[int index] { get; }
	}

	public interface IStructuralFeatureCollection: ICollection
	{
		IStructuralFeature GetItemByName(string name);
		IStructuralFeature this[int index] { get; }
	}

	public interface IMethodCollection: ICollection
	{
		IMethod GetItemByName(string name);
		IMethod this[int index] { get; }
	}

	public interface IAttributeCollection: ICollection
	{
		IAttribute GetItemByName(string name);
		IAttribute this[int index] { get; }
	}

	public interface ICollectionKinds
	{
		ICollectionType CollectionType { get; }
		ISetType SetType { get; }
		IBagType BagType { get; }
		ISequenceType SequenceType { get; }
		IOrderedSetType OrderedSetType { get; }
	}

	public interface IModelElement
	{
		Boolean IsSame(IModelElement value);
		int GetIsSameHashCode();
		String Name { get; }
		IConstraintCollection Constraints { get; }
		ITaggedValueCollection TaggedValues { get; }
		IPackage Package { get; }
	}

	public interface IEcoPredefinedTypes
	{
		IDataType Byte { get; }
		IDataType SByte { get; }
		IDataType Int16 { get; }
		IDataType UInt16 { get; }
		IDataType Int32 { get; }
		IDataType Int64 { get; }
		IDataType Single { get; }
		IDataType Double { get; }
		IDataType Boolean { get; }
		IDataType Char { get; }
		IDataType Decimal { get; }
		IDataType String_ { get; }
		IDataType DateTime { get; }
		IDataType TimeSpan { get; }
		IClass NullType { get; }
		IClass RootClassType { get; }
		IEnumerationType OclSortDirection { get; }
		ICollectionType GenericCollectionType { get; }
	}

	public interface IEcoTypeSystem: IModelElement
	{
		IClassifier GetClassifierByName(string name);
		IClassifier GetClassifierByType(System.Type t);
		IClass GetClassByType(System.Type t);
		bool IsRunnable { get; }
		bool IsBrowsable { get; }
		string NonRunnableReason { get; }
		string[] Errors { get;}
		IClassCollection AllClasses { get; }
		IClassifierCollection AllDatatypes { get; }
		IAssociationCollection AllAssociations { get; }
		IPackageCollection Packages { get; }
		IEcoPredefinedTypes PredefinedTypes { get; }
		string RegionDefinitions { get; }
		bool GenerateDefaultRegions { get; }
	}

	public interface IPackage: IModelElement
	{
		IModelElementCollection ownedElements { get; }
		IEcoPackage EcoPackage { get; }
	}

	public interface IEcoPackage
	{
		IClassCollection Classes { get; }
		IAssociationCollection Associations { get; }
		IPackageCollection Packages { get; }
		string Id { get; }
	}

	public interface IConstraint: IModelElement
	{
		IExpression Body { get; }
		IEcoConstraint EcoConstraint { get; }
	}

	public interface IEcoConstraint
	{
		bool IsAutoGenerated { get; }
		string Description { get; }
	}

	public interface ITaggedValue
	{
		string Tag { get; }
		string Value { get; }
	}

	public interface IStereotype: IModelElement
	{
	}

	public interface IQualifier: IModelElement
	{
		IClassifier Type_ { get; }
	}

	public interface IClassifier: IModelElement
	{
		System.Type ObjectType { get; }
		IClassifierCollection SubTypes { get; }
		IClassifierCollection SuperTypes { get; }
		IFeatureCollection Features { get; }
		ClassifierType ClassifierType { get; }
		IEcoClassifier EcoClassifier { get; }
		ICollectionType CollectionType { get; }
	}

	public interface IEcoClassifier
	{
		IClassifier LeastCommonType(IClassifier classifier);
		bool ConformsTo(IClassifier classifier);
		System.Object ConvertAsObject(object value, IClassifier toType);
		bool IsA(IClassifier classifier);
		IEcoTypeSystem TypeSystem { get; }
		bool IsAbstract { get; }
	}

	public interface IClass: IClassifier
	{
		IEcoClass EcoClass { get; }
		bool IsAssociationClass { get; }
		IAssociation Association { get; }
		IStateMachine StateMachine { get; }
	}

	public enum OptimisticLockingMode
	{
		Off,
		ModifiedMembers,
		AllMembers,
		TimeStamp
	};

	public interface IEcoClass: IEcoClassifier
	{
		string DefaultStringRepresentation { get; }
		bool Versioned { get; }
		IStructuralFeatureCollection AllStructuralFeatures { get; }
		IMethodCollection AllMethods { get; }
		int FirstOwnStructuralFeatureIndex { get; }
		IClass SuperClass { get; }
		bool Persistent { get; }
		bool IsLinkClass { get; }
		int InternalIndex { get; }
		bool GenerateDefaultRegion { get; }
		OptimisticLockingMode OptimisticLockingMode { get; }
		string GetDerivationOcl(IStructuralFeature feature);
		string Id { get; }
	}

	public interface IAssociation: IModelElement
	{
		IAssociationEnd this[int index] { get; }
		int Count { get; }
		IClass AssociationClass {get; }
		IEcoAssociation EcoAssociation { get; }
	}

	public interface IEcoAssociation
	{
		string Id { get; }
	}



	public interface IFeature: IModelElement
	{
		ScopeKind OwnerScope { get; }
		VisibilityKind Visibility { get; }
		IClassifier Owner { get; }
		FeatureType FeatureType { get; }
		IEcoFeature EcoFeature { get; }
	}

	public interface IEcoFeature
	{
		string Id { get; }
	}

	public interface IStructuralFeature: IFeature
	{
		IMultiplicity Multiplicity { get; }
		ChangeableKind Changeability	 { get; }
		IClassifier Type_ { get; }
		IEcoStructuralFeature EcoStructuralFeature { get; }
	}

	public interface IEcoStructuralFeature: IEcoFeature
	{
		int IndexInAllStructuralFeatures { get; }
		bool IsStoredInObject { get; }
		bool Persistent { get; }
		bool IsDerived { get; }
		bool IsReverseDerived { get; }
		bool IsDelayedFetch { get; }
		bool CanHaveOldValue { get; }
		bool HasUserCode { get; }
		bool EncouragesOptimisticLockingOnDeletedOnly { get; }
		int LoopbackIndex { get; }
		SaveAction SaveAction { get; }
		MethodInfo DeriveAndSubscribeMethod { get; }
		PropertyInfo PropertyInfo { get; }
	}

	public interface IMultiplicity
	{
		int Lower { get; }
		int Upper { get; }
		bool IsUnbounded { get; }
		bool IsSingle { get; }
	}

	public interface IAttribute: IStructuralFeature
	{
		IExpression InitialValue { get; }
		IEcoAttribute EcoAttribute { get; }
	}

	public interface IEcoAttribute: IEcoStructuralFeature
	{
		bool AllowNull { get; }
		string DefaultDbValue { get; }
		int Length { get; }
		object InitialValueAsObject { get; }
		bool IsStateAttribute { get; }
	}

	public interface IAssociationEnd: IStructuralFeature
	{
		bool IsNavigable { get; }
		OrderingKind Ordering { get; }
		AggregationKind Aggregation { get; }
		IAssociationEnd OppositeEnd { get; }
		IAssociation Association {get; }
		IQualifierCollection Qualifiers { get; }
		AssociationEndKind AssociationEndKind { get; }
		IEcoAssociationEnd EcoAssociationEnd { get; }
	}

	public enum AssociationEndDefaultRegionMode {Default, None, Cascade, IndependentCascade, Existence};

	public interface IEcoAssociationEnd: IEcoStructuralFeature
	{
		IClass Class_ { get; }
		IClass AssociationClass { get; }
		IAssociationEnd AssociationClassEndPointingHere { get; }
		IAssociationEnd AssociationClassEndPointingAway { get; }
		IAssociationEnd EndPointingToAssociationClass { get; }
		DeleteAction DeleteAction { get; }
		AssociationEndDefaultRegionMode DefaultRegionMode { get; }
		IAssociationEnd MainAssociationEnd { get; }
	}

	public interface IMethod: IFeature
	{
		IParameterCollection Parameters { get; }
		IClassifier ReturnType { get; }
		Boolean IsQuery { get; }
		IEcoMethod EcoMethod { get; }
	}

	public interface IEcoMethod: IEcoFeature
	{
		MethodInfo MethodInfo { get; }
		Boolean IsTrigger { get; }
		Boolean IsEcoCallable { get; }
		Boolean IsImplementedAsActivity { get; }
		IActivity Implementation { get; }
		IExpression Precondition { get; }
		IMethodTrigger Trigger { get; }
	}


	public interface IParameter: IModelElement
	{
		ParameterDirectionKind Kind { get; }
		IClassifier Type { get; }
	}

	public interface IExpression
	{
		string Language { get; }
		string Body { get; }
	}

	public interface IDataType: IClassifier
	{
	}

	public interface ITupleType: IDataType
	{
	}

	public interface ISimpleType: IDataType
	{
		IEcoSimpleType EcoSimpleType { get; }
	}

	public interface IEcoSimpleType: IEcoClassifier
	{
		System.Object DefaultContent { get; }
		System.Object AsObjectFromString(string value);
	}

	public interface IPrimitiveType: ISimpleType
	{
		IEcoPrimitiveType EcoPrimitiveType { get; }
	}

	public interface IEcoPrimitiveType: IEcoSimpleType
	{
		System.Object AssignConvertAsObject(object value, IClassifier toType);
	}

	public interface IEnumerationType: ISimpleType
	{
		IEnumerationLiteralCollection Literals { get; }
	}

	public interface IEnumerationLiteral: IModelElement
	{
		IEcoEnumerationLiteral EcoEnumerationLiteral { get; }
	}

	public interface IEcoEnumerationLiteral
	{
		int IntegerValue { get; }
		object EnumValue { get; }
	}

	public interface ICollectionType: IDataType
	{
		IClassifier ElementType { get; }
		CollectionTypeKind Kind { get; }
		ICollectionKinds CollectionKinds { get; }
	}

	public interface IBagType: ICollectionType
	{
	}

	public interface IOrderedSetType: ICollectionType
	{
	}

	public interface ISequenceType: ICollectionType
	{
	}

	public interface ISetType: ICollectionType
	{
	}

	// State Machine

	public interface IRegionCollection: ICollection
	{
		IRegion GetItemByName(string name);
		IRegion this[int index] { get; }
	}

	public interface ITransitionCollection: ICollection
	{
		ITransition this[int index] { get; }
	}

	public interface IVertexCollection: ICollection
	{
		IVertex GetItemByName(string name);
		IVertex this[int index] { get; }
	}

	public interface IStateCollection: ICollection
	{
		IState this[int index] { get; }
		bool Contains(object value);
	}

	public interface ITriggerCollection: ICollection
	{
		ITrigger this[int index] { get; }
	}

	public interface IStateNamedCollection: IStateCollection
	{
		IState GetItemByName(string name);
	}

	public interface IRegionContainer
	{
		IRegionCollection Regions { get; }
		IEcoRegionContainer EcoRegionContainer { get; }
	}

	public interface IEcoRegionContainer
	{
		string FullStatemachineName { get; }
		bool Contains(IState state);
	}

	public interface IStateMachine: IModelElement, IRegionContainer
	{
		IClass Class_ { get; }
		IEcoStateMachine EcoStateMachine { get; }
	}

	public interface IEcoStateMachine: IEcoRegionContainer
	{
		IVertex GetVertexByPartialName(string name);
		IVertex GetVertexByFullName(string name);
		IRegion GetRegionByPartialName(string name);
		IRegion GetRegionByFullName(string name);
		ITrigger GetTriggerByName(string name);
		IAttributeCollection AllStateAttributes { get; }
	}

	public interface IRegion: IModelElement
	{
		ITransitionCollection Transitions { get; }
		IVertexCollection Vertexes { get; }
		IRegionContainer Container { get; }
		IEcoRegion EcoRegion { get; }
		string FullStatemachineName { get; }
	}

	public interface IEcoRegion
	{
		IAttribute StateAttribute { get; }
		IStateNamedCollection States { get; }
		IPseudostate InitialState { get; }
		IStateCollection AllStates { get; }
		bool IsTopLevel { get; }
	}

	public interface ITransition
	{
		IRegion Region { get; }
		IExpression Guard { get; }
		IVertex Source { get; }
		IVertex Target { get; }
		ITrigger Trigger { get; }
		IActivity Effect { get; }
	}

	public interface IActivity
	{
		string Language { get; }
		string Body { get; }
	}

	public interface IVertex: IModelElement
	{
		ITransitionCollection Outgoing { get; }
		ITransitionCollection Incoming { get; }
		IRegion Container { get; }
		string FullStatemachineName { get; }
		IEcoVertex EcoVertex { get; }
	}

	public interface IEcoVertex
	{
		bool HasUntriggeredOutgoing { get; }
		bool IsFinalState { get; }
		IRegion TopSimpleRegion { get; }
	}

	public enum PseudostateKind
	{
		Initial
	}

	public interface IPseudostate: IVertex
	{
		PseudostateKind Kind { get; }
	}

	public interface IState: IVertex, IRegionContainer
	{
		IActivity Entry { get; }
		IActivity Exit { get; }
		bool IsSimple { get; }
		bool IsComposite { get; }
		bool IsOrthogonal { get; }
		IEcoState EcoState { get; }
	}

	public interface IStateRepresentationPart
	{
		IAttribute Attribute { get; }
		object Value { get; }
	}

	public interface IStateRepresentationPartCollection: ICollection
	{
		IStateRepresentationPart this[int index] { get; }
	}

	public interface IEcoState: IEcoVertex, IEcoRegionContainer
	{
		IStateRepresentationPartCollection Representation { get; }
	}

	public interface IFinalState:IState
	{
	}

	public interface ITrigger
	{
		string Name { get; }
	}

	public interface IMethodTrigger: ITrigger
	{
		IMethod Method { get; }
	}

	public interface ITypeConverter
	{
		System.Object Convert(System.Object value);
	}

	public abstract class UmlTaggedValues
	{
		public class BooleanTV
		{
			public readonly string Tag;

			public BooleanTV(string tag)
			{
				Tag = tag;
			}

			public bool GetValue(ITaggedValueCollection taggedValues, bool defaultValue)
			{
				return AsBool(taggedValues, Tag, defaultValue);
			}

			public bool GetValue(ITaggedValueCollection taggedValues)
			{
				return AsBool(taggedValues, Tag);
			}
		}

		public class StringTV
		{
			public readonly string Tag;

			public StringTV(string tag)
			{
				Tag = tag;
			}

			///<exception cref="ArgumentNullException">Thrown if <paramref name="taggedValues"/> is null.</exception>
			public string GetValue(ITaggedValueCollection taggedValues, string defaultValue)
			{
				if (taggedValues == null)
					throw new ArgumentNullException("taggedValues"); // Do not localize
				return taggedValues.ValueForTag(Tag, defaultValue);
			}

			///<exception cref="ArgumentNullException">Thrown if <paramref name="taggedValues"/> is null.</exception>
			public string GetValue(ITaggedValueCollection taggedValues)
			{
				if (taggedValues == null)
					throw new ArgumentNullException("taggedValues"); // Do not localize
				return taggedValues.ValueForTag(Tag);
			}
		}

		public static readonly StringTV Documentation = new StringTV("documentation");
		public static readonly BooleanTV Derived = new BooleanTV("derived");

		public const string TrueString = "True"; // change to const when dccil fixed
		public const string FalseString = "False"; // change to const when dccil fixed

		public abstract class TaggedEnum
		{
			public static string FormatValues(string[] values)
			{
				bool first = true;
				StringBuilder sb = new StringBuilder();
				sb.Append("["); // do not localize

				foreach (string s in values)
				{
					if (!first)
						sb.Append(", "); // do not localize
					first = false;
					sb.Append(s);
				}
				sb.Append("]"); // do not localize

				return sb.ToString();
			}
		}

		///<summary>
		///If the element is persistent or transient
		///</summary>
		public abstract class Persistence: TaggedEnum
		{
			public const string Tag = "persistence";
			public const string Persistent = "persistent";
			public const string Transient = "transient";
			public static readonly string[] AllValues = new string[] {Persistent, Transient};

			///<exception cref="ArgumentNullException">Thrown if <paramref name="taggedValues"/> is null.</exception>
			///<exception cref="ArgumentOutOfRangeException">Thrown if tagged value is not one of the legal values in <see cref="Persistence"/>.</exception>
			public static bool GetIsPersistent(ITaggedValueCollection taggedValues)
			{
				if (taggedValues == null)
					throw new ArgumentNullException("taggedValues"); // Do not localize
				ITaggedValue tv = taggedValues.GetItemByTag(Tag);
				if (StrEqual(tv.Value, Persistence.Persistent))
					return true;
				if (StrEqual(tv.Value, Persistence.Transient))
					return false;
				throw new ArgumentOutOfRangeException(InterfacesStringRes.sEnumValueNotInRange(Tag, FormatValues(AllValues)));
			}
		}

		protected static bool StrEqual(string s1, string s2)
		{
			if (s1 == s2)
				return true;
			else
				return System.String.Compare(s1, s2, true, CultureInfo.InvariantCulture) == 0;
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="taggedValues"/> is null.</exception>
		public static bool AsBool(ITaggedValueCollection taggedValues, string tag, bool defaultValue)
		{
			if (taggedValues == null)
				throw new ArgumentNullException("taggedValues"); // Do not localize
			ITaggedValue tv = taggedValues.GetItemByTag(tag);
			if (tv == null)
				return defaultValue;
			return AsBool(tv.Value);
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="taggedValues"/> is null.</exception>
		///<exception cref="ArgumentOutOfRangeException">Thrown if tagged value is not available.</exception>
		public static bool AsBool(ITaggedValueCollection taggedValues, string tag)
		{
			if (taggedValues == null)
				throw new ArgumentNullException("taggedValues"); // Do not localize
			ITaggedValue tv = taggedValues.GetItemByTag(tag);
			if (tv == null)
				throw new ArgumentOutOfRangeException("tag"); // Do not localise
			return AsBool(tv.Value);
		}

		///<exception cref="ArgumentOutOfRangeException">Thrown if <paramref name="value"/> does not represent a boolean string.</exception>
		public static bool AsBool(string value)
		{
			if (StrEqual(value, TrueString))
				return true;
			if (StrEqual(value, FalseString))
				return false;
			throw new ArgumentOutOfRangeException(InterfacesStringRes.sBooleanValueMustBe(value, TrueString, FalseString));
		}

		public static string BoolAsString(bool value)
		{
			if (value)
				return TrueString;
			else
				return FalseString;
		}
	}

	public abstract class EcoTaggedValues: UmlTaggedValues
	{
		public const string EcoPrefix = "Eco.";
		public const string Name = "<Name>";
		public const string Default_Value = "<Default>";

		///<summary>
		///How the object should handle optimistic locking
		///</summary>
		public abstract class OptimisticLocking: TaggedEnum
		{
			public const string Tag = EcoPrefix + "OptimisticLocking";
			public const string Default = Default_Value;
			public const string Off = "Off";
			public const string ModifiedMembers = "ModifiedMembers";
			public const string AllMembers = "AllMembers";
			public const string TimeStamp = "TimeStamp";
			public static readonly string[] AllValues = new string[] {Default, Off, ModifiedMembers, AllMembers, TimeStamp};

			///<exception cref="ArgumentNullException">Thrown if <paramref name="taggedValues"/> is null.</exception>
			///<exception cref="ArgumentOutOfRangeException">Thrown if tagged value is not one of the legal values in <see cref="OptimisticLockingMode"/>.</exception>
			public static OptimisticLockingMode GetValue(ITaggedValueCollection taggedValues, OptimisticLockingMode defaultValue )
			{
				if (taggedValues == null)
					throw new ArgumentNullException("taggedValues"); // Do not localize
				string value = taggedValues.ValueForTag(Tag, Default);
				if (StrEqual(value, Default))
					return defaultValue;
				if (StrEqual(value, Off))
					return OptimisticLockingMode.Off;
				if (StrEqual(value, ModifiedMembers))
					return OptimisticLockingMode.ModifiedMembers;
				if (StrEqual(value, AllMembers))
					return OptimisticLockingMode.AllMembers;
				if (StrEqual(value, TimeStamp))
					return OptimisticLockingMode.TimeStamp;
				throw new ArgumentOutOfRangeException(InterfacesStringRes.sEnumValueNotInRange(Tag, FormatValues(AllValues)));
			}
		}

		///<summary>
		///Determines how to treat the association end when the owning object is deleted
		///</summary>
		public abstract class DeleteAction: TaggedEnum
		{
			public const string Tag = EcoPrefix + "DeleteAction";
			public const string Default = Default_Value;
			public const string Allow = "Allow";
			public const string Prohibit = "Prohibit";
			public const string Cascade = "Cascade";
			public static readonly string[] AllValues = new string[] {Default, Allow, Prohibit, Cascade};

			///<exception cref="ArgumentNullException">Thrown if <paramref name="taggedValues"/> is null.</exception>
			///<exception cref="ArgumentOutOfRangeException">Thrown if tagged value is not one of the legal values in <see cref="DeleteAction"/>.</exception>
			public static UmlRt.DeleteAction GetValue(ITaggedValueCollection taggedValues, UmlRt.DeleteAction defaultValue )
			{
				if (taggedValues == null)
					throw new ArgumentNullException("taggedValues"); // Do not localize
				string value = taggedValues.ValueForTag(Tag, Default);
				if (StrEqual(value, Default))
					return defaultValue;
				if (StrEqual(value, Allow))
					return UmlRt.DeleteAction.Allow;
				if (StrEqual(value, Prohibit))
					return UmlRt.DeleteAction.Prohibit;
				if (StrEqual(value, Cascade))
					return UmlRt.DeleteAction.Cascade;
				throw new ArgumentOutOfRangeException(InterfacesStringRes.sEnumValueNotInRange(Tag, FormatValues(AllValues)));
			}
		}

		///<summary>
		///How a member should be handled when written to the database
		///</summary>
		public abstract class SaveAction: TaggedEnum
		{
			public const string Tag = EcoPrefix + "SaveAction";
			public const string Freeze = "Freeze";
			public const string DbAssign = "DbAssign";
			public const string None = "None";
			public static readonly string[] AllValues = new string[] {Freeze, DbAssign, None};

			///<exception cref="ArgumentNullException">Thrown if <paramref name="taggedValues"/> is null.</exception>
			///<exception cref="ArgumentOutOfRangeException">Thrown if tagged value is not one of the legal values in <see cref="SaveAction"/>.</exception>
			public static UmlRt.SaveAction GetValue(ITaggedValueCollection taggedValues)
			{
				if (taggedValues == null)
					throw new ArgumentNullException("taggedValues"); // Do not localize
				string value = taggedValues.ValueForTag(Tag);
				if (StrEqual(value, Freeze))
					return UmlRt.SaveAction.Freeze;
				if (StrEqual(value, DbAssign))
					return UmlRt.SaveAction.DbAssign;
				if (StrEqual(value, None))
					return UmlRt.SaveAction.None;
				throw new ArgumentOutOfRangeException(InterfacesStringRes.sEnumValueNotInRange(Tag, FormatValues(AllValues)));
			}
		}

		///<summary>
		/// Determines how default regions are to be exanded across an association
		///</summary>
		public abstract class DefaultRegionMode: TaggedEnum
		{
			public const string Tag = EcoPrefix + "DefaultRegionMode";
			public const string Default = "Default";
			public const string None = "None";
			public const string Existence = "Existence";
			public const string IndependentCascade = "IndependentCascade";
			public const string Cascade = "Cascade";
			public static readonly string[] AllValues = new string[] {Default, None, Existence, IndependentCascade, Cascade};

			///<exception cref="ArgumentNullException">Thrown if <paramref name="taggedValues"/> is null.</exception>
			///<exception cref="ArgumentOutOfRangeException">Thrown if tagged value is not one of the legal values in <see cref="AssociationEndDefaultRegionMode"/>.</exception>
			public static AssociationEndDefaultRegionMode GetValue(ITaggedValueCollection taggedValues, AssociationEndDefaultRegionMode defaultValue )
			{
				if (taggedValues == null)
					throw new ArgumentNullException("taggedValues"); // Do not localize
				string value = taggedValues.ValueForTag(Tag, Default);
				if (StrEqual(value, Default))
					return defaultValue;
				if (StrEqual(value, None))
					return AssociationEndDefaultRegionMode.None;
				if (StrEqual(value, Existence))
					return AssociationEndDefaultRegionMode.Existence;
				if (StrEqual(value, Cascade))
					return AssociationEndDefaultRegionMode.Cascade;
				if (StrEqual(value, IndependentCascade))
					return AssociationEndDefaultRegionMode.IndependentCascade;
				throw new ArgumentOutOfRangeException(InterfacesStringRes.sEnumValueNotInRange(Tag, FormatValues(AllValues)));
			}
		}

		///<summary>
		///Controls in what table to map instances of a class
		///</summary>
		public abstract class TableMapping: TaggedEnum
		{
			public const string Tag = EcoPrefix + "TableMapping";
			public const string Own = "Own";
			public const string Parent = "Parent";
			public const string Children = "Children";
			public const string Imported = "Imported";
			public static readonly string[] AllValues = new string[] {Own, Parent, Children, Imported};

			///<exception cref="ArgumentNullException">Thrown if <paramref name="taggedValues"/> is null.</exception>
			///<exception cref="ArgumentOutOfRangeException">Thrown if tagged value is not one of the legal values in <see cref="TableMapping"/>.</exception>
			public static UmlRt.TableMapping GetValue(ITaggedValueCollection taggedValues)
			{
				if (taggedValues == null)
					throw new ArgumentNullException("taggedValues"); // Do not localize
				string value = taggedValues.ValueForTag(Tag);
				if (StrEqual(value, Own))
					return UmlRt.TableMapping.Own;
				if (StrEqual(value, Parent))
					return UmlRt.TableMapping.Parent;
				if (StrEqual(value, Children))
					return UmlRt.TableMapping.Children;
				if (StrEqual(value, Imported))
					return UmlRt.TableMapping.Imported;
				throw new ArgumentOutOfRangeException(InterfacesStringRes.sEnumValueNotInRange(Tag, FormatValues(AllValues)));
			}
		}

		public static readonly StringTV TableName = new StringTV(EcoPrefix + "TableName");
		public static readonly StringTV PMapperName = new StringTV(EcoPrefix + "PMapper");
		public static readonly StringTV DefaultStringRepresentation = new StringTV(EcoPrefix + "DefaultStringRepresentation");
		public static readonly StringTV DerivationExpressions = new StringTV(EcoPrefix + "DerivationExpressions");
		public static readonly StringTV FormerNames = new StringTV(EcoPrefix + "FormerNames");
		public static readonly StringTV RegionDefinitions = new StringTV(EcoPrefix + "RegionDefinitions");
		public static readonly StringTV Length = new StringTV(EcoPrefix + "Length");
		public static readonly StringTV ColumnName = new StringTV(EcoPrefix + "ColumnName");
		public static readonly StringTV DerivationOCL = new StringTV(EcoPrefix + "DerivationOCL");
		public static readonly StringTV DefaultDBValue = new StringTV(EcoPrefix + "DefaultDBValue");
		public static readonly StringTV StateAttribute = new StringTV(EcoPrefix + "StateAttribute");
		public static readonly StringTV Representation = new StringTV(EcoPrefix + "Representation");

		public static readonly BooleanTV Versioned = new BooleanTV(EcoPrefix + "Versioned");
		public static readonly BooleanTV GenerateDefaultRegions = new BooleanTV(EcoPrefix + "GenerateDefaultRegions");
		public static readonly BooleanTV GenerateDefaultRegion = new BooleanTV(EcoPrefix + "GenerateDefaultRegion");
		public static readonly BooleanTV GenerateMultiplicityConstraints = new BooleanTV(EcoPrefix + "GenerateMultiplicityConstraints");
		public static readonly BooleanTV AllowNULL = new BooleanTV(EcoPrefix + "AllowNULL");
		public static readonly BooleanTV DelayedFetch = new BooleanTV(EcoPrefix + "DelayedFetch");
		public static readonly BooleanTV DerivedSettable = new BooleanTV(EcoPrefix + "DerivedSettable");
		public static readonly BooleanTV Embed = new BooleanTV(EcoPrefix + "Embed");
		public static readonly BooleanTV IsStateAttribute = new BooleanTV(EcoPrefix + "IsStateAttribute");
		public static readonly BooleanTV IsTrigger = new BooleanTV(EcoPrefix + "IsTrigger");
		public static readonly BooleanTV Ordered = new BooleanTV(EcoPrefix + "Ordered");
		public static readonly StringTV DatabaseName = new StringTV(EcoPrefix + "Database");

		public static readonly StringTV Body = new StringTV(EcoPrefix + "Body");
		public static readonly StringTV Precondition = new StringTV(EcoPrefix + "Precondition");
	}
}
